/************************************************************************
 * NAME:	fs-hdos.c
 *
 * DESCR:	Implements the Heathkit HDOS file system on top of the floppy struct.
 *
 *
 * NOTES:	
 ************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include "getopt.h"
#include "fs-utility.h"
#include "standard.h"
#include "floppy.h"
#include "fs-coco.h"


/************************************************************************
 * NAME:	fs_coco_init() and fs_coco_cleanup()
 *
 * DESCR:	Initializes/cleans-up the COCO file system structures.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_coco_init(struct floppy *floppy, struct fs_coco *fs)
{
    extern int floppy_read_sector(struct floppy *, int, int, int, char *);
    extern int floppy_write_sector(struct floppy *, int, int, int, char *);

    if (!coco_init(&fs->cocofs,floppy,floppy_read_sector,floppy_write_sector)) {
	return(FALSE);
    }

    return (coco_settings(&fs->cocofs,1,5));
}

int
fs_coco_cleanup(struct fs_coco *fs)
{
    return(coco_cleanup(&fs->cocofs));
}

/************************************************************************
 * NAME:	fs_coco_report()
 *
 * DESCR:	Generates a report on the filesystem structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_coco_report(struct fs_coco *fs, int verbosity)
{
    char	buffer[1000];

    if (verbosity == 2) {
    }

    coco_dir_report(&fs->cocofs,verbosity);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_coco_add()
 *
 * DESCR:	Adds the referenced file to the COCO image.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_coco_add(struct fs_coco *fs, char *name, int unusedfd)
{
    struct coco_file	*file;
    unsigned char	 buffer[BUFSIZ];
    int			 i;
    coco_inode		 inode;
    int			 fd;
    struct stat		 statbuf;
    int			 needed_sectors;
    int			 filetype = COCO_FILETYPE_MACHINE;	/* default type	*/
    int			 ascii_flag = COCO_ASCII_YES;

    /* check free space first	*/

    fd = open(name,O_RDONLY);
    if (fd < 0) {
	return(FALSE);
    }

    fstat(fd,&statbuf);


    {	    /* this subsection sets the filetype in some fashion	*/

	char	base[9];	/* both 1 extra for null on the end	*/
	char	ext[4];

	struct { 
	    char	*ext;
	    int		filetype;
	} *fptr, filetype_table[] = {
	                 "BAS", COCO_FILETYPE_BASIC,
			 "BIN", COCO_FILETYPE_MACHINE,
			 "MAC", COCO_FILETYPE_MACHINE,
			 "DAT", COCO_FILETYPE_DATA,
			 "TXT", COCO_FILETYPE_TEXT,
			 0, 0 };

	fs_filename_split(fs_basename(name),base,ext,TRUE);

	for (fptr = filetype_table; fptr->ext != 0; fptr++) {
	    if (strcasecmp(ext,fptr->ext) == 0) {
		filetype = fptr->filetype;
		break;
	    }
	}
    }

    {	/* this subsection sets the ascii flag on the file	*/

	int	i;
	int	count;

	if ( (count = read(fd,buffer,BUFSIZ)) < 0)
	    return(FALSE);

	for (i=0; i < count; i++) {
	    if (buffer[i] > 128) {
		ascii_flag = COCO_ASCII_NO;
	    }
	}

	lseek(fd,(off_t)0,SEEK_SET);
    }
		
    if (statbuf.st_size == 0) {
	needed_sectors = 1;
    } else {
	needed_sectors = (statbuf.st_size - 1) / COCO_SECTOR_SIZE + 1;
    }

    if (needed_sectors <= coco_fat_free_grans(&fs->cocofs)*COCO_SECTORS_PER_GRAN ) {

	inode = coco_dir_find(&fs->cocofs,name);

	if (inode == COCO_INODE_NULL) {

	    file = coco_file_new(&fs->cocofs,fs_basename(name),filetype,ascii_flag);

	    if (file != NULL) {
		while ( i = read(fd,buffer,BUFSIZ) ) {
		    if (!coco_file_write(file,buffer,i)) {
			break;
		    }
		}

		coco_file_close(file);

		if (coco_dir_write(&fs->cocofs)) {
		    if (coco_fat_write(&fs->cocofs)) {
			close(fd);
			return(TRUE);
		    }
		}
	    }
	}
    }
    close(fd);
    return(FALSE);
}

/************************************************************************
 * NAME:	fs_coco_del()
 *
 * DESCR:	Deletes the named file from the filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_coco_del(struct fs_coco *fs, char *name)
{
    coco_inode	inode;

    inode = coco_dir_find(&fs->cocofs,fs_basename(name));

    if (inode == COCO_INODE_NULL) {
	return(FALSE);
    }

    coco_dir_filedel(&fs->cocofs,inode);

    if (!coco_dir_write(&fs->cocofs)) {
	return(FALSE);
    }
    if (!coco_fat_write(&fs->cocofs)) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_coco_extract()
 *
 * DESCR:	Extracts the named file from the filesystem, sending it
 *		out
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_coco_extract(struct fs_coco *fs, char *name, int fd)
{
    struct coco_file	*file;
    char		 buffer[BUFSIZ];
    int			 output;
    int			 i;
    coco_inode		 inode;

    inode = coco_dir_find(&fs->cocofs,fs_basename(name));

    if (inode == COCO_INODE_NULL) {
	return(FALSE);
    }

    if ((output = open(name,O_CREAT|O_WRONLY|O_TRUNC,0777)) < 0) {
	return(FALSE);
    }

    file = coco_file_open(&fs->cocofs,fs_basename(name));

    if (file == NULL) {
	return(FALSE);
    }

    while ( i = coco_file_read(file,buffer,BUFSIZ) ) {
	if (i < 0) {
	    return(FALSE);
	}
	write(output,buffer,i);
    }

    coco_file_close(file);
    close(output);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_coco_description()
 *
 * DESCR:	Returns the description of this type of file system.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
fs_coco_description(void)
{
    return("TRS-80 Color Computer (CoCo)");
}


/************************************************************************
 * NAME:	fs_coco_compress()
 *
 * DESCR:
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
fs_coco_compress(struct fs_coco *fs)
{
}
